/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include "config_handler.h"
#include "miscellaneous.h"
#include <QFile>
#include <QMessageBox>
#include <QDebug>

/* This function reads the config.txt file and saves the provided parameters into variables to be used in the program. */

QT_USE_NAMESPACE
ConfigHandler::ConfigHandler(QString filename)
{
	ReadConfig(filename);
}

QString ConfigHandler::GetConfigValue(QString val)
{
	return configmap.value(val);
}

void ConfigHandler::ReadConfig(QString filename)
{
	QFile file(filename);
	if(!file.open(QIODevice::ReadOnly))
	{
		QMessageBox message;
		message.critical(nullptr, "Configuration File unavailable!", file.errorString() + "\n" + file.fileName());
		exit(-1);
	}
	else
	{
		/* Read the file and save values to configmap */
		while(!file.atEnd())
		{
			QString line;
			line = file.readLine();
			line.replace("\t","");	//remove tabs;
			line.replace("\r","");	//remove \r\n at the end;
			line.replace("\n","");	//remove \r\n at the end;

			if (!(line.startsWith("#") || line.isEmpty()))
			{
//				 qDebug() << line;
                 QStringList list = line.split(QRegExp("\\s*=\\s*"));
				 if(list.count() == 2)
				 {
					configmap.insert(list.at(0), list.at(1));
				 }
				 else
				 {
					qDebug() << "line syntax incorrect: " << list << "ignored line";
				 }
			}
		}
	}
	file.close();
}


/* -----------------------------------------------------------------------------------------------------*/
/* Read name of serial ports																			*/
/* -----------------------------------------------------------------------------------------------------*/

//
// TODO:
// all these ifs are totally unnecessary... the strings should get properly stored immediately, the gets should just be simple gets...
//

int ConfigHandler::get_channel_count()
{
	if (configmap.contains("channel_count")){
		channel_count = configmap.value("channel_count").toInt();
	}
	return channel_count;
}

/* For single-channel setups, use this select the channel number */
QString ConfigHandler::get_channel_select()
{
	if (configmap.contains("channel_select"))
	{
		channel_select = configmap.value("channel_select");
	}
	return channel_select;
}

QString ConfigHandler::get_output_port()
{
	if (configmap.contains("output_port")){
		output_port = configmap.value("output_port");
	}
	return output_port;
}

QString ConfigHandler::get_SGX_port(int intrasys_num)
{
	SGX_port = "";
	if (configmap.contains("SGX_port_" + QString::number(intrasys_num)))
	{
		SGX_port = configmap.value("SGX_port_" + QString::number(intrasys_num));
	}
//	else
//	{
//		qDebug() << "Subsystem[" << intrasys_num << "]: ConfigHandler ->" << "SGX_port_" + QString::number(intrasys_num) + " not defined in config.";
//	}

	return SGX_port;
}


/* -----------------------------------------------------------------------------------------------------*/
/* Remote Command Mode Settings */
/* -----------------------------------------------------------------------------------------------------*/
bool ConfigHandler::get_support_RCM_mode_USB_live()
{
	if (configmap.contains("support_RCM_mode_USB_live")){
		support_RCM_mode_USB_live = (bool)QString(configmap.value("support_RCM_mode_USB_live")).toInt();
		//qDebug() << "support_RCM_mode_USB_live: " << support_RCM_mode_USB_live;
	}
	return support_RCM_mode_USB_live;
}

bool ConfigHandler::get_support_RCM_mode_USB_blind()
{
	if (configmap.contains("support_RCM_mode_USB_blind")){
		support_RCM_mode_USB_blind = (bool)QString(configmap.value("support_RCM_mode_USB_blind")).toInt();
		//qDebug() << "support_RCM_mode_USB_blind: " << support_RCM_mode_USB_blind;
	}
	return support_RCM_mode_USB_blind;
}

bool ConfigHandler::get_support_RCM_mode_RS232_live()
{
	if (configmap.contains("support_RCM_mode_RS232_live")){
		support_RCM_mode_RS232_live = (bool)QString(configmap.value("support_RCM_mode_RS232_live")).toInt();
		//qDebug() << "support_RCM_mode_RS232_live: " << support_RCM_mode_RS232_live;
	}
	return support_RCM_mode_RS232_live;
}

bool ConfigHandler::get_support_RCM_mode_RS232_blind()
{
	if (configmap.contains("support_RCM_mode_RS232_blind")){
		support_RCM_mode_RS232_blind = (bool)QString(configmap.value("support_RCM_mode_RS232_blind")).toInt();
		//qDebug() << "support_RCM_mode_RS232_blind: " << support_RCM_mode_RS232_blind;
	}
	return support_RCM_mode_RS232_blind;
}

//bool ConfigHandler::get_support_RCM_mode_RS485_live()
//{
//	if (configmap.contains("support_RCM_mode_RS485_live")){
//		support_RCM_mode_RS485_live = (bool)QString(configmap.value("support_RCM_mode_RS485_live")).toInt();
//		//qDebug() << "support_RCM_mode_RS485_live: " << support_RCM_mode_RS485_live;
//	}
//	return support_RCM_mode_RS485_live;
//}

bool ConfigHandler::get_support_RCM_mode_RS485_blind()
{
	if (configmap.contains("support_RCM_mode_RS485_blind")){
		support_RCM_mode_RS485_blind = (bool)QString(configmap.value("support_RCM_mode_RS485_blind")).toInt();
		//qDebug() << "support_RCM_mode_RS485_blind: " << support_RCM_mode_RS485_blind;
	}
	return support_RCM_mode_RS485_blind;
}

bool ConfigHandler::get_support_RCM_mode_TCP_live()
{
	if (configmap.contains("support_RCM_mode_TCP_live")){
		support_RCM_mode_TCP_live = (bool)QString(configmap.value("support_RCM_mode_TCP_live")).toInt();
		//qDebug() << "support_RCM_mode_TCP_live: " << support_RCM_mode_TCP_live;
	}
	return support_RCM_mode_TCP_live;
}

bool ConfigHandler::get_support_RCM_mode_TCP_blind()
{
	if (configmap.contains("support_RCM_mode_TCP_blind")){
		support_RCM_mode_TCP_blind = (bool)QString(configmap.value("support_RCM_mode_TCP_blind")).toInt();
		//qDebug() << "support_RCM_mode_TCP_blind: " << support_RCM_mode_TCP_blind;
	}
	return support_RCM_mode_TCP_blind;
}

QString ConfigHandler::get_RCM_port_USB()
{
	if (configmap.contains("RCM_port_USB")){
		RCM_port_USB = configmap.value("RCM_port_USB");
	}
	return RCM_port_USB;
}

QString ConfigHandler::get_RCM_port_RS232()
{
	if (configmap.contains("RCM_port_RS232")){
		RCM_port_RS232 = configmap.value("RCM_port_RS232");
	}
	return RCM_port_RS232;
}

QString ConfigHandler::get_RCM_port_RS485()
{
	if (configmap.contains("RCM_port_RS485")){
		RCM_port_RS485 = configmap.value("RCM_port_RS485");
	}
	return RCM_port_RS485;
}

int ConfigHandler::get_RCM_port_TCP()
{
	if (configmap.contains("RCM_port_TCP")){
		RCM_port_TCP = configmap.value("RCM_port_TCP").toInt();
	}
	return RCM_port_TCP;
}

/* -----------------------------------------------------------------------------------------------------*/
/* Read System stuff																					*/
/* -----------------------------------------------------------------------------------------------------*/
bool ConfigHandler::get_cursor_enabled()
{
	if (configmap.contains("cursor_enabled")){
		cursor_enabled = (bool)configmap.value("cursor_enabled").toInt();
		//qDebug() << "cursor_enabled: " << cursor_enabled;
	}
	return cursor_enabled;
}

bool ConfigHandler::get_fullscreen_enabled()
{
	if (configmap.contains("fullscreen_enabled")){
		fullscreen_enabled = (bool)QString(configmap.value("fullscreen_enabled")).toInt();
		//qDebug() << "fullscreen_enabled: " << fullscreen_enabled;
	}
	return fullscreen_enabled;
}

int ConfigHandler::get_polling_rate_data()
{
	if (configmap.contains("polling_rate_data")){
		polling_rate_data = QString(configmap.value("polling_rate_data")).toInt();
		//qDebug() << "polling_rate_data: " << polling_rate_data << "ms";
	}
	return polling_rate_data;
}

bool ConfigHandler::get_continuous_readings_enabled()
{
	/* Switches between non-stop readings and readings only while RF is enabled.
	 * Purpose: if a bug breaks coms, have the possibility to turn off the automated stuff for easier debugging...*/
	if (configmap.contains("continuous_readings_enabled")){
		continuous_readings_enabled = (bool)QString(configmap.value("continuous_readings_enabled")).toInt();
		//qDebug() << "continuous_readings_enabled: " << continuous_readings_enabled;
	}
	return continuous_readings_enabled;
}

bool ConfigHandler::get_console_output_enabled()
{
	if (configmap.contains("console_output_enabled")){
		console_output_enabled = (bool)QString(configmap.value("console_output_enabled")).toInt();
		//qDebug() << "console_output_enabled: " << console_output_enabled;
	}
	return console_output_enabled;
}

bool ConfigHandler::get_reset_on_startup()
{
	if 	(configmap.contains("reset_on_startup")){
		reset_on_startup = (bool)QString(configmap.value("reset_on_startup")).toInt();
		//qDebug() << "reset_on_startup: " << reset_on_startup;
	}
	return reset_on_startup;
}

bool ConfigHandler::get_init_failed_exit()
{
	if (configmap.contains("init_failed_exit")){
		init_failed_exit = (bool)QString(configmap.value("init_failed_exit")).toInt();
		//qDebug() << "init_failed_exit: " << init_failed_exit;
	}
	return init_failed_exit;
}

int ConfigHandler::get_read_temperature()
{
	if (configmap.contains("read_temperature")){
		read_temperature = QString(configmap.value("read_temperature")).toInt();
		//qDebug() << "read_temperature: " << read_temperature;
	}
	return read_temperature;
}

double ConfigHandler::get_read_temperature_SG()
{
	if (configmap.contains("read_temperature_SG")){
		read_temperature = QString(configmap.value("read_temperature_SG")).toDouble();
		//qDebug() << "read_temperature_SG: " << read_temperature_SG;
	}
	return read_temperature;
}

bool ConfigHandler::get_read_power()
{
	if (configmap.contains("read_power"))
	{
		read_power = QString(configmap.value("read_power")).toInt();
		//qDebug() << "read_power: " << read_power;

	}
	return read_power;
}

int ConfigHandler::get_error_clear_mode()
{
	if (configmap.contains("error_clear_mode")){
		error_clear_mode = configmap.value("error_clear_mode").toInt();
		return error_clear_mode;
	}
	return error_clear_mode;
}


/* Configure if the power readings should be compensated for the PWM duty cycle */
/* Older SGx firmware returns peak powers only, newer firmware has compensation built in. */
bool ConfigHandler::get_duty_cycle_compensation_enabled()
{
	if (configmap.contains("duty_cycle_compensation_enabled"))
	{
		duty_cycle_compensation_enabled = (bool)(QString(configmap.value("duty_cycle_compensation_enabled")).toInt());
		//qDebug() << "duty_cycle_compensation_enabled: " << duty_cycle_compensation_enabled;
	}
	return duty_cycle_compensation_enabled;
}

bool ConfigHandler::get_support_DVGA()
{
	if (configmap.contains("support_DVGA")){
		support_DVGA = (bool)(QString(configmap.value("support_DVGA")).toInt());
		//qDebug() << "support_DVGA: " << support_DVGA;
	}
	return support_DVGA;
}


//
// TODO:
// This OK signal pin config item is a bit vague. It's a sort of pseudo interlock thing but never quite got applied correctly.
// Figure out a better name / implementation for it.
//
int ConfigHandler::get_ok_signal_pin()
{
	if (configmap.contains("ok_signal_pin")){
		ok_signal_pin = configmap.value("ok_signal_pin").toInt();
		//qDebug() << "ok_signal_pin: " << ok_signal_pin;
	}
	return ok_signal_pin;
}

bool ConfigHandler::get_ODC_workaround_enabled()
{
	if (configmap.contains("ODC_workaround_enabled")){
		ODC_workaround_enabled = (bool)QString(configmap.value("ODC_workaround_enabled")).toInt();
		//qDebug() << "ODC_workaround_enabled: " << ODC_workaround_enabled;
	}
	return ODC_workaround_enabled;
}

QString ConfigHandler::get_rotary_switch_pins()
{
	if (configmap.contains("rotary_switch_pins")){
		rotary_switch_pins = configmap.value("rotary_switch_pins");
		//qDebug() << "rotary_switch_pins: " << rotary_switch_pins;
	}
	return rotary_switch_pins;
}

bool ConfigHandler::get_support_HAT_B14_0835()
{
	if (configmap.contains("support_HAT_B14_0835")){
		support_HAT_B14_0835 = (bool)QString(configmap.value("support_HAT_B14_0835")).toInt();
		//qDebug() << "support_HAT_B14_0835: " << support_HAT_B14_0835;
	}
	return support_HAT_B14_0835;
}

bool ConfigHandler::get_enforce_software_limits_frequency()
{
	if (configmap.contains("enforce_software_limits_frequency")){
		enforce_software_limits_frequency = (bool)QString(configmap.value("enforce_software_limits_frequency")).toInt();
		//qDebug() << "enforce_software_limits_frequency: " << enforce_software_limits_frequency;
	}
	return enforce_software_limits_frequency;
}

bool ConfigHandler::get_enforce_software_limits_power()
{
	if (configmap.contains("enforce_software_limits_power")){
		enforce_software_limits_power = (bool)QString(configmap.value("enforce_software_limits_power")).toInt();
		//qDebug() << "enforce_software_limits_power: " << enforce_software_limits_power;
	}
	return enforce_software_limits_power;
}

bool ConfigHandler::get_enforce_software_limits_autocorrect()
{
	if (configmap.contains("enforce_software_limits_autocorrect")){
		enforce_software_limits_autocorrect = (bool)QString(configmap.value("enforce_software_limits_autocorrect")).toInt();
		//qDebug() << "enforce_software_limits_autocorrect: " << enforce_software_limits_autocorrect;
	}
	return enforce_software_limits_autocorrect;
}

//
// TODO:
// Home menu power control management
// These should only be temporary and can be deleted later when an automated solution is provided.
//
int ConfigHandler::get_feedforward_PWRS_controls_enabled()
{
	if (configmap.contains("feedforward_PWRS_controls_enabled")){
		feedforward_PWRS_controls_enabled = configmap.value("feedforward_PWRS_controls_enabled").toDouble();
		//qDebug() << "feedforward_PWRS_controls_enabled: " << feedforward_PWRS_controls_enabled;
	}
	return feedforward_PWRS_controls_enabled;
}

int ConfigHandler::get_feedforward_mode()
{
	if (configmap.contains("feedforward_mode")){
		feedforward_mode = configmap.value("feedforward_mode").toDouble();
		//qDebug() << "feedforward_mode: " << feedforward_mode;
	}
	return feedforward_mode;
}

bool ConfigHandler::get_support_negative_attenuation()
{
	if (configmap.contains("support_negative_attenuation")){
		support_negative_attenuation = (bool)QString(configmap.value("support_negative_attenuation")).toInt();
		//qDebug() << "support_negative_attenuation: " << support_negative_attenuation;
	}
	return support_negative_attenuation;
}

bool ConfigHandler::get_use_runtime_based_reset_detection()
{
	if (configmap.contains("use_runtime_based_reset_detection")){
		use_runtime_based_reset_detection = (bool)QString(configmap.value("use_runtime_based_reset_detection")).toInt();
		//qDebug() << "use_runtime_based_reset_detection: " << use_runtime_based_reset_detection;
	}
	return use_runtime_based_reset_detection;
}

bool ConfigHandler::get_report_PA_type()
{
	if (configmap.contains("report_PA_type")){
		report_PA_type = (bool)QString(configmap.value("report_PA_type")).toInt();
		//qDebug() << "report_PA_type: " << report_PA_type;
	}
	return report_PA_type;
}

/* -----------------------------------------------------------------------------------------------------*/
/* GPIOs																								*/
/* -----------------------------------------------------------------------------------------------------*/
int ConfigHandler::get_GPIO_3v3_out()
{
	if (configmap.contains("GPIO_3v3_out")){
		GPIO_3v3_out = configmap.value("GPIO_3v3_out").toDouble();
		//qDebug() << "GPIO_3v3_out: " << GPIO_3v3_out;
	}
	return GPIO_3v3_out;
}

int ConfigHandler::get_GPIO_interlock_in()
{
	if (configmap.contains("GPIO_interlock_in")){
		GPIO_interlock_in = configmap.value("GPIO_interlock_in").toInt();
		//qDebug() << "GPIO_interlock_in: " << GPIO_interlock_in;
	}
	return GPIO_interlock_in;
}


/* -----------------------------------------------------------------------------------------------------*/
/* Read menus and features																				*/
/* -----------------------------------------------------------------------------------------------------*/
bool ConfigHandler::get_menu_home_enabled()
{
	if (configmap.contains("menu_home_enabled")){
		menu_home_enabled = (bool)QString(configmap.value("menu_home_enabled")).toInt();
		//qDebug() << "menu_home_enabled: " << menu_home_enabled;
	}
	return menu_home_enabled;
}

bool ConfigHandler::get_menu_splitter_enabled()
{
	if (configmap.contains("menu_splitter_enabled")){
		menu_splitter_enabled = (bool)QString(configmap.value("menu_splitter_enabled")).toInt();
		//qDebug() << "menu_splitter_enabled: " << menu_splitter_enabled;
	}
	return menu_splitter_enabled;
}

bool ConfigHandler::get_menu_ALL_enabled()
{
	if (configmap.contains("menu_ALL_enabled")){
		menu_ALL_enabled = (bool)QString(configmap.value("menu_ALL_enabled")).toInt();
		//qDebug() << "menu_ALL_enabled: " << menu_ALL_enabled;
	}
	return menu_ALL_enabled;
}

bool ConfigHandler::get_menu_DLL_enabled()
{
	if (configmap.contains("menu_DLL_enabled")){
		menu_DLL_enabled = (bool)QString(configmap.value("menu_DLL_enabled")).toInt();
		//qDebug() << "menu_DLL_enabled: " << menu_DLL_enabled;
	}
	return menu_DLL_enabled;
}

bool ConfigHandler::get_menu_sweep_enabled()
{
	if (configmap.contains("menu_sweep_enabled")){
		menu_sweep_enabled = (bool)QString(configmap.value("menu_sweep_enabled")).toInt();
		//qDebug() << "menu_sweep_enabled: " << menu_sweep_enabled;
	}
	return menu_sweep_enabled;
}

bool ConfigHandler::get_menu_PSU_enabled()
{
	if (configmap.contains("menu_PSU_enabled")){
		menu_PSU_enabled = (bool)QString(configmap.value("menu_PSU_enabled")).toInt();
		//qDebug() << "menu_PSU_enabled: " << menu_PSU_enabled;
	}
	return menu_PSU_enabled;
}

bool ConfigHandler::get_menu_help_enabled()
{
	if (configmap.contains("menu_help_enabled")){
		menu_help_enabled = (bool)QString(configmap.value("menu_help_enabled")).toInt();
		//qDebug() << "menu_help_enabled: " << menu_help_enabled;
	}
	return menu_help_enabled;
}

bool ConfigHandler::get_menu_settings_enabled()
{
	if (configmap.contains("menu_settings_enabled")){
		menu_settings_enabled = (bool)QString(configmap.value("menu_settings_enabled")).toInt();
		//qDebug() << "menu_settings_enabled: " << menu_settings_enabled;
	}
	return menu_settings_enabled;
}

bool ConfigHandler::get_menu_power_enabled()
{
	if (configmap.contains("menu_power_enabled")){
		menu_power_enabled = (bool)QString(configmap.value("menu_power_enabled")).toInt();
		//qDebug() << "menu_power_enabled: " << menu_power_enabled;
	}
	return menu_power_enabled;
}

bool ConfigHandler::get_menu_PID_enabled()
{
	if (configmap.contains("menu_PID_enabled")){
		menu_PID_enabled = (bool)QString(configmap.value("menu_PID_enabled")).toInt();
		//qDebug() << "menu_home_enabled: " << menu_home_enabled;
	}
	return menu_PID_enabled;
}

bool ConfigHandler::get_support_phase_control()
{
	if (configmap.contains("support_phase_control")){
		support_phase_control = (bool)QString(configmap.value("support_phase_control")).toInt();
		//qDebug() << "support_phase_control: " << support_phase_control;
	}
	return support_phase_control;
}

bool ConfigHandler::get_support_power_offset()
{
	if (configmap.contains("support_power_offset")){
		support_power_offset = (bool)QString(configmap.value("support_power_offset")).toInt();
		//qDebug() << "support_power_offset: " << support_power_offset;
	}
	return support_power_offset;
}

bool ConfigHandler::get_support_PWM_triggering()
{
	if (configmap.contains("support_PWM_triggering")){
		support_PWM_triggering = (bool)QString(configmap.value("support_PWM_triggering")).toInt();
		//qDebug() << "support_PWM_triggering: " << support_PWM_triggering;
	}
	return support_PWM_triggering;
}

bool ConfigHandler::get_support_coherency_standalone()
{
	if (configmap.contains("support_coherency_standalone")){
		support_coherency_standalone = (bool)QString(configmap.value("support_coherency_standalone")).toInt();
		//qDebug() << "support_coherency_standalone: " << support_coherency_standalone;
	}
	return support_coherency_standalone;
}

bool ConfigHandler::get_support_coherency_LVDS_master()
{
	if (configmap.contains("support_coherency_LVDS_master")){
		support_coherency_LVDS_master = (bool)QString(configmap.value("support_coherency_LVDS_master")).toInt();
		//qDebug() << "support_coherency_LVDS_master: " << support_coherency_LVDS_master;
	}
	return support_coherency_LVDS_master;
}

bool ConfigHandler::get_support_coherency_LVDS_slave_end()
{
	if (configmap.contains("support_coherency_LVDS_slave_end")){
		support_coherency_LVDS_slave_end = (bool)QString(configmap.value("support_coherency_LVDS_slave_end")).toInt();
		//qDebug() << "support_coherency_LVDS_slave_end: " << support_coherency_LVDS_slave_end;
    }
	return support_coherency_LVDS_slave_end;
}

bool ConfigHandler::get_support_coherency_LVDS_slave_inline()
{
	if (configmap.contains("support_coherency_LVDS_slave_inline")){
		support_coherency_LVDS_slave_inline = (bool)QString(configmap.value("support_coherency_LVDS_slave_inline")).toInt();
		//qDebug() << "support_coherency_LVDS_slave_inline: " << support_coherency_LVDS_slave_inline;
	}
	return support_coherency_LVDS_slave_inline;
}

bool ConfigHandler::get_support_coherency_AUX_master()
{
	if (configmap.contains("support_coherency_AUX_master")){
		support_coherency_AUX_master= (bool)QString(configmap.value("support_coherency_AUX_master")).toInt();
		//qDebug() << "support_coherency_AUX_master: " << support_coherency_AUX_master;
	}
	return support_coherency_AUX_master;
}

bool ConfigHandler::get_support_coherency_AUX_slave_end()
{
	if (configmap.contains("support_coherency_AUX_slave_end")){
		support_coherency_AUX_slave_end = (bool)QString(configmap.value("support_coherency_AUX_slave_end")).toInt();
		//qDebug() << "support_coherency_AUX_slave_end: " << support_coherency_AUX_slave_end;
	}
	return support_coherency_AUX_slave_end;
}

bool ConfigHandler::get_support_external_triggering_mode()
{
	if (configmap.contains("support_external_triggering_mode")){
		support_external_triggering_mode = (bool)QString(configmap.value("support_external_triggering_mode")).toInt();
		//qDebug() << "support_external_triggering_mode: " << support_external_triggering_mode;
	}
	return support_external_triggering_mode;
}

int ConfigHandler::get_external_triggering_synchronization_enable()
{
	if (configmap.contains("external_triggering_synchronization_enable")){
		external_triggering_synchronization_enable = QString(configmap.value("external_triggering_synchronization_enable")).toInt();
		//qDebug() << "external_triggering_synchronization_enable: " << external_triggering_synchronization_enable;
	}
	return external_triggering_synchronization_enable;
}

int ConfigHandler::get_external_triggering_synchronization_delay_us()
{
	if (configmap.contains("external_triggering_synchronization_delay_us")){
		external_triggering_synchronization_delay_us = QString(configmap.value("external_triggering_synchronization_delay_us")).toInt();
		//qDebug() << "external_triggering_synchronization_delay_us: " << external_triggering_synchronization_delay_us;
	}
	return external_triggering_synchronization_delay_us;
}


bool ConfigHandler::get_support_AIS_mode()
{
	if (configmap.contains("support_AIS_mode")){
		support_AIS_mode = (bool)QString(configmap.value("support_AIS_mode")).toInt();
		//qDebug() << "support_AIS_mode: " << support_AIS_mode;
	}
	return support_AIS_mode;
}

bool ConfigHandler::get_support_modbus_mode()
{
	if (configmap.contains("support_modbus_mode")){
		support_modbus_mode = (bool)QString(configmap.value("support_modbus_mode")).toInt();
		//qDebug() << "support_modbus_mode: " << support_modbus_mode;
	}
	return support_modbus_mode;
}

bool ConfigHandler::get_support_ethercat_mode()
{
	if (configmap.contains("support_ethercat_mode")){
		support_ethercat_mode = (bool)QString(configmap.value("support_ethercat_mode")).toInt();
	}
	return support_ethercat_mode;
}

bool ConfigHandler::get_support_logging()
{
	if (configmap.contains("support_logging")){
		support_logging = (bool)QString(configmap.value("support_logging")).toInt();
		//qDebug() << "support_logging: " << support_logging;
	}
	return support_logging;
}

bool ConfigHandler::get_support_feedforward_control()
{
	if (configmap.contains("support_feedforward_control")){
		support_feedforward_control = (bool)QString(configmap.value("support_feedforward_control")).toInt();
		//qDebug() << "support_feedforward_control: " << support_feedforward_control;
	}
	return support_feedforward_control;
}

int ConfigHandler::get_support_amplifier_mode()
{
	if (configmap.contains("support_amplifier_mode")){
		support_amplifier_mode = QString(configmap.value("support_amplifier_mode")).toInt();
		//qDebug() << "support_amplifier_mode: " << support_amplifier_mode;
	}
	return support_amplifier_mode;
}

bool ConfigHandler::get_support_PID_control()
{
	if (configmap.contains("support_PID_control")){
		support_PID_control = (bool)QString(configmap.value("support_PID_control")).toInt();
		//qDebug() << "support_PID_control: " << support_PID_control;
	}
	return support_PID_control;
}

bool ConfigHandler::get_support_power_control_modes()
{
	if (configmap.contains("support_power_control_modes")){
		support_power_control_modes = (bool)QString(configmap.value("support_power_control_modes")).toInt();
		//qDebug() << "support_power_control_modes: " << support_power_control_modes;
	}
	return support_power_control_modes;
}

bool ConfigHandler::get_controls_PWM_enabled()
{
	if (configmap.contains("controls_PWM_enabled")){
		controls_PWM_enabled = (bool)QString(configmap.value("controls_PWM_enabled")).toInt();
		//qDebug() << "controls_PWM_enabled: " << controls_PWM_enabled;
	}
	return controls_PWM_enabled;
}

bool ConfigHandler::get_controls_CW_enabled()
{
	if (configmap.contains("controls_CW_enabled")){
		controls_CW_enabled = (bool)QString(configmap.value("controls_CW_enabled")).toInt();
		//qDebug() << "controls_CW_enabled: " << controls_CW_enabled;
	}
	return controls_CW_enabled;
}

bool ConfigHandler::get_controls_ALL_enabled()
{
	if (configmap.contains("controls_ALL_enabled")){
		controls_ALL_enabled = (bool)QString(configmap.value("controls_ALL_enabled")).toInt();
		//qDebug() << "controls_ALL_enabled: " << controls_ALL_enabled;
	}
	return controls_ALL_enabled;
}

bool ConfigHandler::get_controls_DLL_enabled()
{
	if (configmap.contains("controls_DLL_enabled")){
		controls_DLL_enabled = (bool)QString(configmap.value("controls_DLL_enabled")).toInt();
		//qDebug() << "controls_DLL_enabled: " << controls_DLL_enabled;
	}
	return controls_DLL_enabled;
}

/* -----------------------------------------------------------------------------------------------------*/
/* RF Splitter board																					*/
/* -----------------------------------------------------------------------------------------------------*/
bool ConfigHandler::get_splitter_channel_disable()
{
	if (configmap.contains("splitter_channel_disable")){
		splitter_channel_disable = QString(configmap.value("splitter_channel_disable")).toInt();
		//qDebug() << "splitter_channel_disable: " << splitter_channel_disable;
	}
	return splitter_channel_disable;
}

int ConfigHandler::get_splitter_channel_count()
{
	if (configmap.contains("splitter_channel_count")){
		splitter_channel_count = QString(configmap.value("splitter_channel_count")).toInt();
		//qDebug() << "splitter_channel_count: " << splitter_channel_count;
	}
	return splitter_channel_count;
}

bool ConfigHandler::get_support_splitter_controls_phase()
{
	if (configmap.contains("support_splitter_controls_phase"))
	{
		support_splitter_controls_phase = QString(configmap.value("support_splitter_controls_phase")).toInt();
		//qDebug() << "support_splitter_controls_phase: " << support_splitter_controls_phase;
	}
	return support_splitter_controls_phase;
}

bool ConfigHandler::get_support_splitter_controls_magnitude()
{
	if (configmap.contains("support_splitter_controls_magnitude"))
	{
		support_splitter_controls_magnitude = QString(configmap.value("support_splitter_controls_magnitude")).toInt();
		//qDebug() << "support_splitter_controls_magnitude: " << support_splitter_controls_magnitude;
	}
	return support_splitter_controls_magnitude;
}

bool ConfigHandler::get_support_splitter_controls_attenuation()
{
	if (configmap.contains("support_splitter_controls_attenuation"))
	{
		support_splitter_controls_attenuation = QString(configmap.value("support_splitter_controls_attenuation")).toInt();
		//qDebug() << "support_splitter_controls_attenuation: " << support_splitter_controls_attenuation;
	}
	return support_splitter_controls_attenuation;
}


/* -----------------------------------------------------------------------------------------------------*/
/* Read important defaults																				*/
/* -----------------------------------------------------------------------------------------------------*/
bool ConfigHandler::get_target_RF_enable()
{
	if (configmap.contains("target_RF_enable")){
		target_RF_enable = (bool)QString(configmap.value("target_RF_enable")).toInt();
		//qDebug() << "target_RF_enable: " << target_RF_enable;
	}
	return target_RF_enable;
}

int ConfigHandler::get_target_PA_type()
{
	if (configmap.contains("target_PA_type")){
		target_PA_type = QString(configmap.value("target_PA_type")).toInt();
		//qDebug() << "target_PA_type: " << target_PA_type;
	}
	return target_PA_type;
}

int ConfigHandler::get_remote_command_mode()
{
	if (configmap.contains("remote_command_mode")){
		target_remote_command_mode = QString(configmap.value("remote_command_mode")).toInt();
		//qDebug() << "remote_command_mode: " << target_remote_command_mode;
	}
	return target_remote_command_mode;
}

int ConfigHandler::get_coherency_mode()
{
	if (configmap.contains("coherency_mode")){
		target_clock_source = QString(configmap.value("coherency_mode")).toInt();
		//qDebug() << "coherency_mode: " << target_coherency_mode;
	}
	return target_clock_source;
}

int ConfigHandler::get_ext_triggering_mode()
{
	if (configmap.contains("ext_triggering_mode")){
		target_ext_trig_mode = QString(configmap.value("ext_triggering_mode")).toInt();
		//qDebug() << "External trigger mode: " << target_ext_trig_mode;
	}
	return target_ext_trig_mode;
}

int ConfigHandler::get_PWM_triggering_mode()
{
	if (configmap.contains("PWM_triggering_mode")){
		target_PWM_trigger_mode = QString(configmap.value("PWM_triggering_mode")).toInt();
		//qDebug() << "PWM_triggering_mode: " << target_PWM_trigger_mode;
	}
	return target_PWM_trigger_mode;
}

bool ConfigHandler::get_default_DVGA_amplifier_forward_enabled()
{
	if (configmap.contains("default_DVGA_amplifier_forward_enabled")){
		default_DVGA_amplifier_forward_enabled = (bool)QString(configmap.value("default_DVGA_amplifier_forward_enabled")).toInt();
		//qDebug() << "default_DVGA_amplifier_forward_enabled: " << default_DVGA_amplifier_forward_enabled;
	}
	return default_DVGA_amplifier_forward_enabled;
}

double ConfigHandler::get_default_DVGA_attenuation_forward()
{
	if (configmap.contains("default_DVGA_attenuation_forward")){
		default_DVGA_attenuation_forward = QString(configmap.value("default_DVGA_attenuation_forward")).toDouble();
		//qDebug() << "default_DVGA_attenuation_forward:" << default_DVGA_attenuation_forward;
	}
	return default_DVGA_attenuation_forward;
}

bool ConfigHandler::get_default_DVGA_amplifier_reflected_enabled()
{
	if (configmap.contains("default_DVGA_amplifier_reflected_enabled")){
		default_DVGA_amplifier_reflected_enabled = (bool)QString(configmap.value("default_DVGA_amplifier_reflected_enabled")).toInt();
		//qDebug() << "default_DVGA_amplifier_reflected_enabled: " << default_DVGA_amplifier_reflected_enabled;
	}
	return default_DVGA_amplifier_reflected_enabled;
}

double ConfigHandler::get_default_DVGA_attenuation_reflected()
{
	if (configmap.contains("default_DVGA_attenuation_reflected")){
		default_DVGA_attenuation_reflected = QString(configmap.value("default_DVGA_attenuation_reflected")).toDouble();
		//qDebug() << "default_DVGA_attenuation_reflected: " << default_DVGA_attenuation_reflected;
	}
	return default_DVGA_attenuation_reflected;
}

double ConfigHandler::get_PWM_minimum_pulse_length_ns()
{
	if (configmap.contains("PWM_minimum_pulse_length_ns")){
		PWM_minimum_pulse_length_ns = QString(configmap.value("PWM_minimum_pulse_length_ns")).toDouble();
		//qDebug() << "PWM_minimum_pulse_length_ns: " << PWM_minimum_pulse_length_ns;
	}
	return PWM_minimum_pulse_length_ns;
}

bool ConfigHandler::get_use_minimum_pulse_length()
{
	if (configmap.contains("use_minimum_pulse_length")){
		use_minimum_pulse_length = (bool)QString(configmap.value("use_minimum_pulse_length")).toInt();
		//qDebug() << "use_minimum_pulse_length: " << use_minimum_pulse_length;
	}
	return use_minimum_pulse_length;
}

bool ConfigHandler::get_modbus_enabled()
{
	if (configmap.contains("modbus_enabled")){
		modbus_enabled = (bool)QString(configmap.value("modbus_enabled")).toInt();
		//qDebug() << "modbus_enabled: " << modbus_enabled;
	}
	return modbus_enabled;
}

bool ConfigHandler::get_ethercat_enabled()
{
	if (configmap.contains("ethercat_enabled")){
		ethercat_enabled = (bool)QString(configmap.value("ethercat_enabled")).toInt();
	}
	return ethercat_enabled;
}

int ConfigHandler::get_logging_rate()
{
	if (configmap.contains("logging_rate")){
		logging_rate = QString(configmap.value("logging_rate")).toInt();
		//qDebug() << "logging_rate: " << logging_rate << "ms";
	}
	return logging_rate;
}

QString ConfigHandler::get_logging_directory()
{
	if (configmap.contains("logging_directory")){
		logging_directory = QString(configmap.value("logging_directory"));
		qDebug() << "logging_directory: " << logging_directory;
	}
	return logging_directory;
}

bool ConfigHandler::get_logging_enabled()
{
	if (configmap.contains("logging_enabled")){
		logging_enabled = (bool)QString(configmap.value("logging_enabled")).toInt();
		//qDebug() << "logging_enabled: " << logging_enabled;
	}
	return logging_enabled;
}


double ConfigHandler::get_FFWD_attenuation_override()
{
	if (configmap.contains("FFWD_attenuation_override")){
		FFWD_attenuation_override = configmap.value("FFWD_attenuation_override").toInt();
		//qDebug() << "FFWD_attenuation_override: " << FFWD_attenuation_override;
	}
	return FFWD_attenuation_override;
}


/* -----------------------------------------------------------------------------------------------------*/
/* Read PSU values																						*/
/* -----------------------------------------------------------------------------------------------------*/
int ConfigHandler::get_PSU_count()
{
	if (configmap.contains("PSU_count")){
		PSU_count = configmap.value("PSU_count").toInt();
		return PSU_count;
	}
	return PSU_count;
}

bool ConfigHandler::get_support_PSU_controls_enable()
{
	if (configmap.contains("support_PSU_controls_enable")){
		support_PSU_controls_enable = configmap.value("support_PSU_controls_enable").toInt();
		return support_PSU_controls_enable;
	}
	return support_PSU_controls_enable;
}

bool ConfigHandler::get_support_PSU_controls_voltage()
{
	if (configmap.contains("support_PSU_controls_voltage")){
		support_PSU_controls_voltage = configmap.value("support_PSU_controls_voltage").toInt();
		return support_PSU_controls_voltage;
	}
	return support_PSU_controls_voltage;
}

bool ConfigHandler::get_support_PSU_controls_current_limit()
{
	if (configmap.contains("support_PSU_controls_current_limit")){
		support_PSU_controls_current_limit = configmap.value("support_PSU_controls_current_limit").toInt();
		return support_PSU_controls_current_limit;
	}
	return support_PSU_controls_current_limit;
}


double ConfigHandler::get_PSU_voltage()
{
	if (configmap.contains("PSU_voltage")){
		PSU_voltage = configmap.value("PSU_voltage").toDouble();
		//qDebug() << "PSU_voltage: " << PSU_voltage;
	}
	return PSU_voltage;
}

double ConfigHandler::get_PSU_current_limit()
{
	if (configmap.contains("PSU_current_limit")){
		PSU_current_limit = configmap.value("PSU_current_limit").toDouble();
		//qDebug() << "PSU_current_limit: " << PSU_current_limit;
	}
	return PSU_current_limit;
}

int ConfigHandler::get_PSU_enabled()
{
	if (configmap.contains("PSU_enabled")){
		PSU_enabled = configmap.value("PSU_enabled").toInt();
		//qDebug() << "PSU_enabled: " << PSU_enabled;
	}
	return PSU_enabled;
}


/* -----------------------------------------------------------------------------------------------------*/
/* Read CW values																						*/
/* -----------------------------------------------------------------------------------------------------*/
double ConfigHandler::get_frequency_min()
{
	if (configmap.contains("frequency_min")){
		frequency_min = QString(configmap.value("frequency_min")).toDouble();
	}
	return frequency_min;
}

double ConfigHandler::get_frequency_max()
{
	if (configmap.contains("frequency_max")){
		frequency_max = QString(configmap.value("frequency_max")).toDouble();
	}
	return frequency_max;
}

double ConfigHandler::get_frequency_value()
{
	if (configmap.contains("frequency_value")){
		freq_value = QString(configmap.value("frequency_value")).toDouble() ;
	}
	return freq_value;
}

double ConfigHandler::get_power_watt_min()
{
	if (configmap.contains("power_watt_min")){
		power_watt_min = QString(configmap.value("power_watt_min")).toDouble();
	}
	return power_watt_min;
}

double ConfigHandler::get_power_watt_max()
{
	if (configmap.contains("power_watt_max")){
		power_watt_max = QString(configmap.value("power_watt_max")).toDouble();
	}
	return power_watt_max;
}

double ConfigHandler::get_power_watt_value()
{
	if (configmap.contains("power_watt_value")){
		power_watt_value = QString(configmap.value("power_watt_value")).toDouble();;
		power_dbm_value = zeroChopper(QString::number(convert_watt_to_dbm(power_watt_value),'f',6)).toDouble();
	}
	return power_watt_value;
}

double ConfigHandler::get_power_dbm_value()
{
	if (configmap.contains("power_watt_value")){
		power_watt_value = QString(configmap.value("power_watt_value")).toDouble();;
		power_dbm_value = zeroChopper(QString::number(convert_watt_to_dbm(power_watt_value),'f',6)).toDouble();
	}
	return power_dbm_value;
}

double ConfigHandler::get_phase_value()
{
	if (configmap.contains("phase_value")){
		phase_value = QString(configmap.value("phase_value")).toDouble();
	}
	return phase_value;
}

/* -----------------------------------------------------------------------------------------------------*/
/* Read PWM values																						*/
/* -----------------------------------------------------------------------------------------------------*/
bool ConfigHandler::get_PWM_enabled()
{
	if (configmap.contains("PWM_enabled")){
		target_PWM_enabled = (bool)QString(configmap.value("PWM_enabled")).toInt();
	}
	return target_PWM_enabled;
}

double ConfigHandler::get_PWM_duty_cycle_value()
{
	if (configmap.contains("PWM_duty_cycle_value")){
		PWM_duty_cycle_value = QString(configmap.value("PWM_duty_cycle_value")).toDouble();
	}
	return PWM_duty_cycle_value;
}

double ConfigHandler::get_PWM_frequency_value()
{
	if (configmap.contains("PWM_frequency_value")){
		PWM_freq_value = QString(configmap.value("PWM_frequency_value")).toDouble();
	}
	return PWM_freq_value;
}

int ConfigHandler::get_PWM_frequency_min()
{
	if (configmap.contains("PWM_frequency_min")){
		PWM_freq_min = QString(configmap.value("PWM_frequency_min")).toInt();
	}
	return PWM_freq_min;
}

int ConfigHandler::get_PWM_frequency_max()
{
	if (configmap.contains("PWM_frequency_max")){
		PWM_freq_max = QString(configmap.value("PWM_frequency_max")).toInt();
	}
	return PWM_freq_max;
}

int ConfigHandler::get_PWM_slave_port()
{
	if (configmap.contains("PWM_slave_port")){
		PWM_slave_port = QString(configmap.value("PWM_slave_port")).toInt();
	}
	return PWM_slave_port;
}

int ConfigHandler::get_PWM_slave_pin()
{
	if (configmap.contains("PWM_slave_pin")){
		PWM_slave_pin = QString(configmap.value("PWM_slave_pin")).toInt();
	}
	return PWM_slave_pin;
}

int ConfigHandler::get_PWM_master_port()
{
	if (configmap.contains("PWM_master_port")){
		PWM_master_port = QString(configmap.value("PWM_master_port")).toInt();
	}
	return PWM_master_port;
}

int ConfigHandler::get_PWM_master_pin()
{
	if (configmap.contains("PWM_master_pin")){
		PWM_master_pin = QString(configmap.value("PWM_master_pin")).toInt();
	}
	return PWM_master_pin;
}

double ConfigHandler::get_PWM_refresh_rate()
{
	if (configmap.contains("PWM_refresh_rate")){
		PWM_refresh_rate = QString(configmap.value("PWM_refresh_rate")).toDouble();
	}
	return PWM_refresh_rate;
}

double ConfigHandler::get_PWM_trigger_pulse()
{
	if (configmap.contains("PWM_trigger_pulse")){
		PWM_trigger_pulse = QString(configmap.value("PWM_trigger_pulse")).toDouble();
	}
	return PWM_trigger_pulse;
}

/* -----------------------------------------------------------------------------------------------------*/
/* Read ALL values																						*/
/* -----------------------------------------------------------------------------------------------------*/
bool ConfigHandler::get_ALL_enabled()
{
	if (configmap.contains("ALL_enabled")){
		target_ALL_enabled = (bool)QString(configmap.value("ALL_enabled")).toInt();
	}
	return target_ALL_enabled;
}

double ConfigHandler::get_ALL_frequency_min()
{
	if (configmap.contains("ALL_frequency_min")){
		ALL_frequency_min = QString(configmap.value("ALL_frequency_min")).toDouble();
	}
	return ALL_frequency_min;
}

double ConfigHandler::get_ALL_frequency_max()
{
	if (configmap.contains("ALL_frequency_max")){
		ALL_frequency_max = QString(configmap.value("ALL_frequency_max")).toDouble();
	}
	return ALL_frequency_max;
}

double ConfigHandler::get_ALL_freqLimit_UP_value()
{
	if (configmap.contains("ALL_upper_freq_limit")){
		ALL_freqLimit_UP_value = QString(configmap.value("ALL_upper_freq_limit")).toDouble();
	}
	return ALL_freqLimit_UP_value;
}

double ConfigHandler::get_ALL_freqLimit_DOWN_value()
{
	if (configmap.contains("ALL_lower_freq_limit")){
		ALL_freqLimit_DOWN_value = QString(configmap.value("ALL_lower_freq_limit")).toDouble();
	}
	return ALL_freqLimit_DOWN_value;
}

double ConfigHandler::get_ALL_threshold_value()
{
	if (configmap.contains("ALL_threshold_db")){
		ALL_threshold_value = QString(configmap.value("ALL_threshold_db")).toDouble();
	}
	return ALL_threshold_value;
}

double ConfigHandler::get_DVGA_forward_attenuation()
{
	if (configmap.contains("DVGA_attenuation_forward")){
		DVGA_attenuation_forward= QString(configmap.value("DVGA_attenuation_forward")).toDouble();
	}
	return DVGA_attenuation_forward;
}

double ConfigHandler::get_DVGA_reflected_attenuation()
{
	if (configmap.contains("DVGA_attenuation_reflected")){
		DVGA_attenuation_reflected= QString(configmap.value("DVGA_attenuation_reflected")).toDouble();
	}
	return DVGA_attenuation_reflected;
}

bool ConfigHandler::get_DVGA_forward_amplifier_enabled()
{
	if (configmap.contains("DVGA_amplifier_forward_enabled")){
		DVGA_amplifier_forward_enabled =(bool)QString(configmap.value("DVGA_amplifier_forward_enabled")).toInt();
	}
	return DVGA_amplifier_forward_enabled;
}

bool ConfigHandler::get_DVGA_reflected_amplifier_enabled()
{
	if (configmap.contains("DVGA_amplifier_reflected_enabled")){
		DVGA_amplifier_reflected_enabled = (bool)QString(configmap.value("DVGA_amplifier_reflected_enabled")).toInt();
	}
	return DVGA_amplifier_reflected_enabled;
}

/* -----------------------------------------------------------------------------------------------------*/
/* Read DLL values																						*/
/* -----------------------------------------------------------------------------------------------------*/
bool ConfigHandler::get_DLL_enabled()
{
	if (configmap.contains("DLL_enabled")){
		target_DLL_enabled = QString(configmap.value("DLL_enabled")).toDouble();
	}
	return target_DLL_enabled;
}

double ConfigHandler::get_DLL_freqLimit_UP()
{
	if (configmap.contains("DLL_upper_freq_limit")){
		DLL_freqLimit_UP_value = QString(configmap.value("DLL_upper_freq_limit")).toDouble();
	}
	return DLL_freqLimit_UP_value;
}

double ConfigHandler::get_DLL_freqLimit_DOWN()
{
	if (configmap.contains("DLL_lower_freq_limit")){
		DLL_freqLimit_DOWN_value = QString(configmap.value("DLL_lower_freq_limit")).toDouble();
	}
	return DLL_freqLimit_DOWN_value;
}

double ConfigHandler::get_DLL_freqLimit_step()
{
	if (configmap.contains("DLL_step_freq_limit")){
		DLL_freqLimit_step = QString(configmap.value("DLL_step_freq_limit")).toDouble();
	}
	return DLL_freqLimit_step;
}

double ConfigHandler::get_DLL_frequency_step()
{
	if (configmap.contains("DLL_step_freq")){
		DLL_freq_step_value = QString(configmap.value("DLL_step_freq")).toDouble();
	}
	return DLL_freq_step_value;
}

double ConfigHandler::get_DLL_frequency_start()
{
	if (configmap.contains("DLL_start_freq")){
		DLL_freq_start_value = QString(configmap.value("DLL_start_freq")).toDouble();
	}
	return DLL_freq_start_value;
}

double ConfigHandler::get_DLL_threshold_db()
{
	if (configmap.contains("DLL_threshold_db")){
		DLL_threshold_value = QString(configmap.value("DLL_threshold_db")).toDouble();
	}
	return DLL_threshold_value;
}

double ConfigHandler::get_DLL_delay_ms()
{

	if (configmap.contains("DLL_delay_ms")){
		DLL_main_delay_value = QString(configmap.value("DLL_delay_ms")).toDouble();
	}
	return DLL_main_delay_value;
}

/* -----------------------------------------------------------------------------------------------------*/
/* Read SWEEP values																					*/
/* -----------------------------------------------------------------------------------------------------*/
double ConfigHandler::get_SWP_start_freq()
{
	if (configmap.contains("SWP_start_freq")){
		SWP_start_freq = QString(configmap.value("SWP_start_freq")).toDouble();
	}
	return SWP_start_freq;
}

double ConfigHandler::get_SWP_stop_freq()
{
	if (configmap.contains("SWP_stop_freq")){
		SWP_stop_freq = QString(configmap.value("SWP_stop_freq")).toDouble();
	}
	return SWP_stop_freq;
}

double ConfigHandler::get_SWP_step_freq()
{
	if (configmap.contains("SWP_step_freq")){
		SWP_step_freq = QString(configmap.value("SWP_step_freq")).toDouble();
	}
	return SWP_step_freq;
}

double ConfigHandler::get_SWP_power_watt()
{
	if (configmap.contains("SWP_power_watt")){
		SWP_power_watt = QString(configmap.value("SWP_power_watt")).toDouble();
		SWP_power_dbm =  zeroChopper(QString::number(convert_watt_to_dbm(SWP_power_watt),'f',6)).toDouble();
	}
	return SWP_power_watt;
}

//Uses SWP_power_watt, but returns dBm value
double ConfigHandler::get_SWP_power_dbm()
{
	if (configmap.contains("SWP_power_watt")){
		SWP_power_watt = QString(configmap.value("SWP_power_watt")).toDouble();
		SWP_power_dbm =  zeroChopper(QString::number(convert_watt_to_dbm(SWP_power_watt),'f',6)).toDouble();
	}
	return SWP_power_dbm;
}

double ConfigHandler::get_SWP_maximum_measure_points()
{
	if (configmap.contains("SWP_maximum_measure_points")){
		SWP_maximum_measure_points = QString(configmap.value("SWP_maximum_measure_points")).toDouble();
	}
	return SWP_maximum_measure_points;
}


/* -----------------------------------------------------------------------------------------------------*/
/* Read Power Control Mode																				*/
/* -----------------------------------------------------------------------------------------------------*/
int ConfigHandler::get_power_control_mode()
{
	if (configmap.contains("power_control_mode")){
		power_control_mode = (int)QString(configmap.value("power_control_mode")).toInt();
		//qDebug() << "power_control_mode: " << power_control_mode;
	}
	return power_control_mode;
}

/* -----------------------------------------------------------------------------------------------------*/
/* Read Analog Input Mode values																		*/
/* -----------------------------------------------------------------------------------------------------*/

double ConfigHandler::get_AIS_attenuation_value()
{
	if (configmap.contains("AIS_attenuation_value")){
		AIS_attenuation_value = QString(configmap.value("AIS_attenuation_value")).toDouble();
		//qDebug() << "AIS_attenuation_value: " << AIS_attenuation_value;
	}
	return AIS_attenuation_value;
}

int ConfigHandler::get_AIS_ADC_max()
{
	if (configmap.contains("AIS_ADC_max")){
		AIS_ADC_max = QString(configmap.value("AIS_ADC_max")).toInt();
		//qDebug() << "AIS_ADC_max: " << AIS_ADC_max;
	}
	return AIS_ADC_max;
}

int ConfigHandler::get_AIS_ADC_min()
{
	if (configmap.contains("AIS_ADC_min")){
		AIS_ADC_min = QString((configmap.value("AIS_ADC_min"))).toInt();
		//qDebug() << "AIS_ADC_min: " << AIS_ADC_min;
	}
	return AIS_ADC_min;
}

double ConfigHandler::get_AIS_attenuation_max_db()
{
	if (configmap.contains("AIS_attenuation_max_db")){
		AIS_attenuation_max_db = QString((configmap.value("AIS_attenuation_max_db"))).toDouble();
		//qDebug() << "AIS_power_min_db: " << AIS_attenuation_max_db;
	}
	return AIS_attenuation_max_db;
}

double ConfigHandler::get_AIS_attenuation_min_db()
{
	if (configmap.contains("AIS_attenuation_min_db")){
		AIS_attenuation_min_db = QString((configmap.value("AIS_attenuation_min_db"))).toDouble();
		//qDebug() << "AIS_power_max_db: " << AIS_attenuation_min_db;
	}
	return AIS_attenuation_min_db;
}

double ConfigHandler::get_AIS_magnitude_max()
{
	if (configmap.contains("AIS_magnitude_max")){
		AIS_magnitude_max = QString((configmap.value("AIS_magnitude_max"))).toDouble();
		//qDebug() << "AIS_magnitude_max: " << AIS_magnitude_max;
	}
	return AIS_magnitude_max;
}

double ConfigHandler::get_AIS_magnitude_min()
{
	if (configmap.contains("AIS_magnitude_min")){
		AIS_magnitude_min = QString((configmap.value("AIS_magnitude_min"))).toDouble();
		//qDebug() << "AIS_magnitude_min: " << AIS_magnitude_min;
	}
	return AIS_magnitude_min;
}


/* -----------------------------------------------------------------------------------------------------*/
/* Read Amplifier Mode values																			*/
/* -----------------------------------------------------------------------------------------------------*/
bool ConfigHandler::get_support_AM_phase_control()
{
	if (configmap.contains("support_AM_phase_control")){
		support_AM_phase_control = (bool)QString((configmap.value("support_AM_phase_control"))).toInt();
		//qDebug() << "support_AM_phase_control: " << support_AM_phase_control;
	}
	return support_AM_phase_control;
}

double ConfigHandler::get_AM_attenuation_max_db()
{
	if (configmap.contains("AM_attenuation_max_db")){
		AM_attenuation_max_db = QString((configmap.value("AM_attenuation_max_db"))).toDouble();
		//qDebug() << "AM_attenuation_max_db: " << AM_attenuation_max_db;
	}
	return AM_attenuation_max_db;
}

double ConfigHandler::get_AM_attenuation_min_db()
{
	if (configmap.contains("AM_attenuation_min_db")){
		AM_attenuation_min_db = QString((configmap.value("AM_attenuation_min_db"))).toDouble();
		//qDebug() << "AM_attenuation_min_db: " << AM_attenuation_min_db;
	}
	return AM_attenuation_min_db;
}

int ConfigHandler::get_AM_controls_enabled()
{
	if (configmap.contains("AM_controls_enabled")){
		AM_controls_enabled = QString((configmap.value("AM_controls_enabled"))).toDouble();
		//qDebug() << "AM_controls_enabled: " << AM_controls_enabled;
	}
	return AM_controls_enabled;
}
